//
//  SceneDelegate.swift
//  Do It
//
//  Created by Jim Dovey on 8/23/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import UIKit
import SwiftUI

// All scenes share this single lazily-initialized data center.
// Previews can create their own.
fileprivate let sharedDataCenter = DataCenter()

class SceneDelegate: UIResponder, UIWindowSceneDelegate {

    var window: UIWindow?

    func scene(_ scene: UIScene, willConnectTo session: UISceneSession, options connectionOptions: UIScene.ConnectionOptions) {
        // Use this method to optionally configure and attach the UIWindow `window` to the provided UIWindowScene `scene`.
        // If using a storyboard, the `window` property will automatically be initialized and attached to the scene.
        // This delegate does not imply the connecting scene or session are new (see `application:configurationForConnectingSceneSession` instead).
        
        // ensure data center etc. are preflighted
        _ = sharedDataCenter
        
        if let response = connectionOptions.notificationResponse {
            if let item = NotificationManager.shared.handleNotificationResponse(response, sharedDataCenter) {
                // show this item in its own scene.
                let view = TodoItemDetail(item: item)
                    .dismissingSceneSession(session)
                presentView(view, in: scene)
                return
            }
        }
        
        // Find a useful activity, if any.
        // START:HandleActivity
        for activity in connectionOptions.userActivities {
            guard let contentID = activity.targetContentIdentifier else {
                continue
            }
            
            switch activity.activityType {
                // END:HandleActivity
                // START:HandleListActivity
            case TodoItemList.activityType:
                guard
                    let uuid = UUID(uuidString: contentID),
                    let list = sharedDataCenter.list(withID: uuid)
                    else { break }
                // START:ListActivityDismiss
                let view = TodoList(list: list)
                    // END:HandleListActivity
                    .dismissingSceneSession(session)
                    // START:HandleListActivity
                presentView(view, in: scene)
                // END:ListActivityDismiss
                return
                // END:HandleListActivity
                
                // START:HandleItemActivity
            case TodoItem.activityType:
                guard
                    let uuid = UUID(uuidString: contentID),
                    let item = sharedDataCenter.item(withID: uuid)
                    else { break }
                let view = TodoItemDetail(item: item)
                    .dismissingSceneSession(session)
                presentView(view, in: scene)
                return
                // END:HandleItemActivity
                
                // START:HandleActivity
            default:
                break
            }
        }
        // END:HandleActivity

        presentView(Home(), in: scene)
    }
    
    private func presentView<Content: View>(_ view: Content, in scene: UIScene) {
        // Use a UIHostingController as window root view controller.
        if let windowScene = scene as? UIWindowScene {
            let idiom = windowScene.traitCollection.userInterfaceIdiom
            let rootView = view
                .environmentObject(sharedDataCenter)
                .environment(\.interfaceIdiom, InterfaceIdiom(idiom))
                .modifier(AssignUndoManager(data: sharedDataCenter))
                .publishingErrors()
            let window = UIWindow(windowScene: windowScene)
            window.rootViewController = KeyCommandHostingController(rootView: rootView)
            self.window = window
            window.makeKeyAndVisible()
        }
    }
    
    private struct AssignUndoManager: ViewModifier {
        var data: DataCenter
        @Environment(\.undoManager) var undoManager
        func body(content: Content) -> some View {
            content.onAppear(perform: self.data.undoManager == nil
                ? { self.data.undoManager = self.undoManager }
                : nil)
        }
    }
}

